#! /usr/bin/python

#    Copyright (c) 2011 David Calle <davidc@framli.eu>
#    Copyright (c) 2011 Michael Hall <mhall119@gmail.com>

#    This program is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License.

#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.

#    You should have received a copy of the GNU General Public License
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os.path
import paramiko
import sys

from gi.repository import GLib, GObject, Gio
from gi.repository import Dee
from gi.repository import Unity

BUS_NAME = "net.launchpad.lens.sshsearch"

SSH_DEFAULT_PORT = '22'
SSHCONFIG = os.path.join('~', '.ssh', 'config')
SSHCONFIG_EXPAND = os.path.expanduser(SSHCONFIG)
KNOWN_HOSTS = os.path.join('~', '.ssh', 'known_hosts')
KNOWN_HOSTS_EXPAND = os.path.expanduser(KNOWN_HOSTS)

TERMINAL_APP = 'gnome-terminal'
TERMINAL_APP_MIMETYPE = 'application-x-desktop'

SCRIPT_DIR = os.path.dirname(os.path.abspath(__file__))
ICON_FILE = os.path.join(SCRIPT_DIR, 'unity-lens-sshsearch.svg')


class Daemon:

    def __init__ (self):
        # The path for the Lens *must* also match the one in our .lens file
        self._lens = Unity.Lens.new("/net/launchpad/lens/sshsearch", "sshsearch")
        self._lens.props.search_hint = "Search SSH connections"
        self._lens.props.visible = True
        self._lens.props.search_in_global = True

        # create the scope
        self._scope = Unity.Scope.new("/net/launchpad/lens/sshsearch/main")
        self._scope.search_in_global = True
        self._scope.connect("search-changed", self.on_search_changed)
        self._scope.connect("filters-changed", self.on_search_changed)
        self._scope.connect("activate-uri", self.on_activate_uri)
        self._scope.export()
        
        # Populate categories
        cats = []
        cats.append(Unity.Category.new(SSHCONFIG,
                                       Gio.ThemedIcon.new(ICON_FILE),
                                       Unity.CategoryRenderer.VERTICAL_TILE))
        cats.append(Unity.Category.new(KNOWN_HOSTS,
                                       Gio.ThemedIcon.new(ICON_FILE),
                                       Unity.CategoryRenderer.VERTICAL_TILE))
        self._lens.props.categories = cats
        
        # Populate filters
        filters = []
        self._lens.props.filters = filters

        self._lens.add_local_scope(self._scope)
        self._lens.export()

        # read/parse ssh-config file
        self._config_hosts = []
        self._config_file = Gio.file_new_for_path(SSHCONFIG_EXPAND)
        self._config_monitor = self._config_file.monitor_file(
            flags=Gio.FileMonitorFlags.NONE, cancellable=None)
        self._config_monitor.connect('changed', self.__read_config)
        self.__read_config(None, self._config_file, None, Gio.FileMonitorEvent.CREATED)

        # read/parse ssh-known_hosts file
        self._known_hosts = []
        self._knownhosts_file = Gio.file_new_for_path(KNOWN_HOSTS_EXPAND)
        self._knownhosts_monitor = self._knownhosts_file.monitor_file(
            flags=Gio.FileMonitorFlags.NONE, cancellable=None)
        self.__read_known_hosts(None, self._knownhosts_file, None, Gio.FileMonitorEvent.CREATED)
        
    def __read_config(self, filemonitor, file, other_file, event_type):
        if not file.query_exists(None):
            self._config_hosts = []
            return

        if (event_type in (Gio.FileMonitorEvent.CREATED,
                           Gio.FileMonitorEvent.CHANGED,
                           Gio.FileMonitorEvent.CHANGES_DONE_HINT,
                           Gio.FileMonitorEvent.ATTRIBUTE_CHANGED)): 
            c = paramiko.SSHConfig()
            c.parse(open(file.get_path()))
            self._config_hosts = [h['host'].lower() for h in c._config if '*' not in h['host']]
        
    def __read_known_hosts(self, filemonitor, file, other_file, event_type):
        if not file.query_exists(None):
            self._known_hosts = []
            return

        if (event_type in (Gio.FileMonitorEvent.CREATED,
                           Gio.FileMonitorEvent.CHANGED,
                           Gio.FileMonitorEvent.CHANGES_DONE_HINT,
                           Gio.FileMonitorEvent.ATTRIBUTE_CHANGED)): 
            h = paramiko.HostKeys(KNOWN_HOSTS_EXPAND)
            self._known_hosts = [host for host in h.keys() if len(host) != 60]

    def on_search_changed(self, scope, search, search_type, cancellable):
        if search:
            search_string = search.props.search_string.lower() or None
        else:
            search_string = None

        if search_type == Unity.SearchType.DEFAULT:
            results = self._scope.props.results_model
        else:
            results = self._scope.props.global_results_model

        self.update_results_model(search_string, results)
        results.flush_revision_queue()
        search.finished()


    def __parse_hoststring(self, hoststring, user):
        # assign host and port
        host = hoststring
        port = SSH_DEFAULT_PORT
        if hoststring.startswith('['):
            host, port = hoststring[1:].split(']:')

        # assign target
        target = host
        if user:
            target = '%s@%s' % (user, host)

        # assign connection-description 
        conn_desc = target
        if port != SSH_DEFAULT_PORT:
            conn_desc = '%s:%s' % (target, port)

        return target, port, conn_desc

    def update_results_model(self, search, model):
        if search is None or search == '':
            return
         
        model.clear()
        searchparts = search.split('@')
        searchhost = searchparts[-1]
        searchuser = ''
        if len(searchparts) == 2:
            searchuser = searchparts[0]
            
        found = [host for host in self._config_hosts if host.find(searchhost) >= 0]
        for host in found:
            target, port, conn_desc = self.__parse_hoststring(host, searchuser)
            model.append('ssh://config/%s/%s' % (searchuser, host),
                         TERMINAL_APP,
                         0,
                         TERMINAL_APP_MIMETYPE,
                         conn_desc, conn_desc, '')
                         
        found = [host for host in self._known_hosts if host.find(searchhost) >= 0]
        for host in found:
            target, port, conn_desc = self.__parse_hoststring(host, searchuser)
            model.append('ssh://known_hosts/%s/%s' % (searchuser, host),
                         TERMINAL_APP,
                         1,
                         TERMINAL_APP_MIMETYPE,
                         conn_desc, conn_desc, '')

    def on_activate_uri (self, scope, uri):
        uri_splitted = uri.split('/')
        hoststring = uri_splitted[-1]
        user = uri_splitted[-2]
        target, port, conn_desc = self.__parse_hoststring(hoststring, user)

        if port == SSH_DEFAULT_PORT:
            # don't call with the port option, because the host definition
            # could be from the ~/.ssh/config file
            GLib.spawn_command_line_async('%s -e "ssh %s"' % (TERMINAL_APP, target))
        else:
            GLib.spawn_command_line_async('%s -e "ssh -p %s %s"' % (TERMINAL_APP, port, target))

        return Unity.ActivationResponse(handled=Unity.HandledType.HIDE_DASH, goto_uri='')


if __name__ == "__main__":
    session_bus_connection = Gio.bus_get_sync(Gio.BusType.SESSION, None)
    session_bus = Gio.DBusProxy.new_sync(session_bus_connection, 0, None,
                                         'org.freedesktop.DBus',
                                         '/org/freedesktop/DBus',
                                         'org.freedesktop.DBus', None)
    result = session_bus.call_sync('RequestName',
                                   GLib.Variant("(su)", (BUS_NAME, 0x4)),
                                   0, -1, None)
                                   
    # Unpack variant response with signature "(u)". 1 means we got it.
    result = result.unpack()[0]
    
    if result != 1 :
        print >> sys.stderr, "Failed to own name %s. Bailing out." % BUS_NAME
        raise SystemExit (1)
    
    daemon = Daemon()
    GObject.MainLoop().run()
